/********************************************************************************************************************
 PIT App Basic Oracle 11g DB Structure Script
 Created By:      VA DEVELOPER
 Date:            01/14/2013
 1stRevision:     Revised to address script errors  and incorporate creation of PIT account
 Date:            01/15/2013 by VA DEVELOPER
 Last Revision:	  Revised to incorporate NSOC vulnerability remediation through explicit invocation of AUTHID clause 
 				  on stored procedure script
 Date:			  01/17/2013 by VA DEVELOPER
 Notes:  This script assumes a database has been established and contains the USERS tablespace and that execution of 
	     scripts are performed with SYSDBA privileges.  The objects created are:
		1. User Account: PIT - limited privileges for creating tables, views, procedures, functions
        2. User Account: PIT_SERVICE - limited privileges for executing ap_PostSurveyRecord()
		3. Table: PIT.VETERANSURVEY - single table used to capture survey record results
		4. Stored Procedure: ap_PostSurveyRecord - performs record insert/update containing parameter values
********************************************************************************************************************/

--==================================================================================================================
--  Create PIT_SERVICE User Acct (be sure to replace pwd characters)
--==================================================================================================================

-- initial create statement

CREATE USER "PIT" IDENTIFIED BY "TEST1";


-- alter to add attributes

ALTER USER "PIT"
DEFAULT TABLESPACE "USERS"
TEMPORARY TABLESPACE "TEMP"
ACCOUNT UNLOCK ;


-- add Roles

GRANT CREATE SESSION TO PIT;
GRANT CREATE VIEW TO PIT;
GRANT CREATE PROCEDURE TO PIT;
GRANT CREATE TABLE TO PIT;

-- add Quota
ALTER USER "PIT" QUOTA 500M ON USERS;

--==================================================================================================================
--  Create PIT_SERVICE User Acct (be sure to replace pwd characters)
--==================================================================================================================


-- initial create statement

CREATE USER "PIT_SERVICE" IDENTIFIED BY "TEST2";

-- alter to add attributes

ALTER USER "PIT_SERVICE"
DEFAULT TABLESPACE "USERS"
TEMPORARY TABLESPACE "TEMP"
ACCOUNT UNLOCK ;

-- add Roles

GRANT CREATE SESSION TO PIT_SERVICE;

--==================================================================================================================
--  Create TABLE PIT.VETERANSURVEY (assumes user PIT created and has schema created)
--==================================================================================================================


DROP TABLE PIT.VETERANSURVEY;

CREATE TABLE PIT.VETERANSURVEY
(
	DEVICEID VARCHAR2(40) NOT NULL,
  SURVEYID VARCHAR2(100) NOT NULL,
	SURVEYDATE DATE NOT NULL,
	FIRSTNAME VARCHAR2(255) NULL,
	MIDDLENAME VARCHAR2(255) NULL,
	LASTNAME VARCHAR2(255) NULL,
	GENDER CHAR(1) NOT NULL,
	AGE NUMBER NULL,
	CHRONICALLYHOMELESS NUMBER NOT NULL ,
	LOCATIONLATITUDE NUMBER NULL,
	LOCATIONLONGITUDE NUMBER NULL,
	LOCATIONACCURACY NUMBER NULL,
	ADDRESS VARCHAR2(255) NULL,
	CITY VARCHAR2(255) NULL,
	STATE VARCHAR2(2) NULL,
	RECORDCREATEDDATE DATE DEFAULT (sysdate) NOT NULL,

    CONSTRAINT CHECK_GENDER CHECK (GENDER IN ('F', 'M', 'f', 'm')) ENABLE,

    CONSTRAINT CHECK_CHRONICALLYHOMELESS CHECK (CHRONICALLYHOMELESS =1 OR CHRONICALLYHOMELESS =0) ENABLE,

    CONSTRAINT PK_SURVEYID PRIMARY KEY (DEVICEID, SURVEYID)
);


--==================================================================================================================
--  Create Stored Procedure PIT.ap_PostSurveyRecord
--==================================================================================================================

create or replace procedure PIT.ap_PostSurveyRecord
(survey_id IN VARCHAR2,
device_id IN VARCHAR2,
survey_date IN DATE,
first_name IN VARCHAR2,
middle_name IN VARCHAR2,
last_name IN VARCHAR2,
veteran_gender IN CHAR,
veteran_age IN NUMBER,
chronically_homeless IN NUMBER,
location_latitude IN NUMBER,
location_longitude IN NUMBER,
location_accuracy IN NUMBER,
location_address IN VARCHAR2,
location_city IN VARCHAR2,
location_state IN VARCHAR2,
returncode OUT VARCHAR2)

-- following clause added for remediation of NSOC Fortify Security Report results [01/17/2013 - KET] 
AUTHID DEFINER AS
/********************************************************************************************************************
 Created By:      VA DEVELOPER
 Date:            01/11/2013
 Last Revision:   Revised to incorporate NSOC vulnerability remediation through explicit invocation of AUTHID clause
 Date:			  01/17/2013 by VA DEVELOPER
*********************************************************************************************************************
 Description:     Designed for the PIT Mobile Application Pilot Initiative.  
                  PIT application stored procedure designed to accept transaction data and perform either insert or 
                  update operation depending on the existence of the primary composite key of Device_ID and Survey_ID.
                  This version created based on the original SP ip_CreateSurveyRecord designed for SQL Server 2008 R2
                  but includes additional feature of update record capability.

 Param notes:   All data explicitly used for transaction - no defaults.  Failure to provide parameterized data results
                in NULL insertion value. The following constraints are applied at the table level for :
                - veteran_gender {M, F}
                - survey_date {valid Oracle-type date or formatted varchar - not required for update statement}
                - chronically_homeless {1, 0}
                Failure to adhere to these constraints will result in transaction failure and returncode presenting 
                the associated Oracle Error Code and Error Message generated from processing the transaction. 

 Precondition:  None.

 Postcondition: Parameterized data is used to create/update record. Output parameter returncode will return '0' if 
                is successful.  Otherwise, the Oracle Error Code and Error Message will be returned as a varchar not 
                to exceed 255 characters.


********************************************************************************************************************/

BEGIN

-- declarations
DECLARE
recordcount NUMBER;     -- used to capture recordcount from query

    BEGIN
    
    recordcount := 0;
    
    -- test for existing record / capture recordcount
    SELECT COUNT(*) INTO recordcount FROM PIT.VETERANSURVEY
       WHERE VETERANSURVEY.SURVEYID = survey_id AND VETERANSURVEY.DEVICEID = device_id;
    
    -- implicit marker for beginning of transaction
    
    -- test for existing record presence
    IF recordcount = 0 THEN
    
       --  no matching record key exists
       --  initiate create statement
       INSERT INTO PIT.VETERANSURVEY
       (SURVEYID, DEVICEID, SURVEYDATE, FIRSTNAME, MIDDLENAME, LASTNAME, GENDER, AGE,
       CHRONICALLYHOMELESS, LOCATIONLATITUDE, LOCATIONLONGITUDE, LOCATIONACCURACY, ADDRESS,
       CITY, STATE)
       VALUES
       (survey_id, device_id, survey_date, first_name, middle_name, last_name, veteran_gender, veteran_age,
       chronically_homeless, location_latitude, location_longitude, location_accuracy, location_address,
       location_city, location_state);
    
       -- commit transaction
       COMMIT;
    
    ELSE
        -- matching record key exists
        -- initiate update statement
        UPDATE PIT.VETERANSURVEY
        SET 
        FIRSTNAME = first_name, 
        MIDDLENAME = middle_name, 
        LASTNAME = last_name,
        GENDER = veteran_gender, 
        AGE = veteran_age,
        CHRONICALLYHOMELESS = chronically_homeless,
        LOCATIONLATITUDE = location_latitude, 
        LOCATIONLONGITUDE = location_longitude,
        LOCATIONACCURACY = location_accuracy, 
        ADDRESS = location_address, 
        CITY = location_city, 
        STATE = location_state
        WHERE SURVEYID = survey_id AND DEVICEID = device_id;
    
        COMMIT;
    
    END IF;
    
    -- commit transaction
    
    -- set return code for successful transaction
    returncode := '0';
    
    -- capture exception
        EXCEPTION
            WHEN OTHERS THEN  -- catching all
                -- set return code to exception text returned
                returncode := SUBSTR(SQLERRM, 1, 255);

                -- rollback transaction
                ROLLBACK;
    END;
END;

/
--==================================================================================================================
--  GRANT EXECUTE for PIT.ap_PostSurveyRecord TO PIT_SERVICE User Acct
--==================================================================================================================
ALTER PROCEDURE PIT.ap_PostSurveyRecord COMPILE; 

GRANT EXECUTE ON PIT.ap_PostSurveyRecord TO "PIT_SERVICE";

/************************************END OF FILE********************************************************************/
